<?php

namespace App\Http\Controllers\Therapists;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Post;
use App\Http\Requests\Therapists\CommunityPosts\CreateRequest;
use App\Http\Resources\Web\CommunityPostsResource;
use App\Http\Resources\Web\PostsResource;
use App\Models\PostComment;
use Excel;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class CommunityPostsController extends Controller
{
    /********* Initialize Permission based Middlewares  ***********/
    public function __construct()
    {
        // $this->middleware('auth');
        // $this->middleware('therapist');
        // $this->middleware('permission:therapist_posts.index');
        $this->middleware(['auth','therapist'],['only' => ['store']]);
        $this->middleware(['auth','therapist'],['only' => ['postLike']]);
        $this->middleware(['auth','therapist'],['only' => ['postComment']]);
        $this->middleware(['auth','therapist'],['only' => ['commentReply']]);
        $this->middleware(['auth','therapist'],['only' => ['import']]);
        // $this->middleware('permission:therapist_posts.update|therapist_posts.is_active',['only' => ['updateStatus']]);
    }

    /********* Getter For Pagination, Searching And Sorting  ***********/
    public function getter($req = null, $export = null)
    {
        $therapist = auth()->user()->therapist;
        if ($req != null) {
            $therapist_posts =  $therapist->therapist_posts()->withAll();
            if ($req->trash && $req->trash == 'with') {
                $therapist_posts =  $therapist_posts->withTrashed();
            }
            if ($req->trash && $req->trash == 'only') {
                $therapist_posts =  $therapist_posts->onlyTrashed();
            }
            if ($req->column && $req->column != null && $req->search != null) {
                $therapist_posts = $therapist_posts->whereLike($req->column, $req->search);
            } else if ($req->search && $req->search != null) {

                $therapist_posts = $therapist_posts->whereLike(['name', 'description'], $req->search);
            }
            if ($req->sort && $req->sort['field'] != null && $req->sort['type'] != null) {
                $therapist_posts = $therapist_posts->OrderBy($req->sort['field'], $req->sort['type']);
            } else {
                $therapist_posts = $therapist_posts->OrderBy('id', 'desc');
            }
            if ($export != null) { // for export do not paginate
                $therapist_posts = $therapist_posts->get();
                return $therapist_posts;
            }
            $totalCommunityPosts = $therapist_posts->count();
            $therapist_posts = $therapist_posts->paginate($req->perPage);
            $therapist_posts = CommunityPostsResource::collection($therapist_posts)->response()->getData(true);

            return $therapist_posts;
        }
        $therapist_posts = CommunityPostsResource::collection($therapist->therapist_posts()->withAll()->orderBy('id', 'desc')->paginate(10))->response()->getData(true);
        return $therapist_posts;
    }

    /********* FETCH ALL CommunityPosts ***********/
    public function index()
    {

        $community_posts =  Post::withAll()->withChildren()->active()->communitPost()->latest()->get();
        $community_posts = CommunityPostsResource::collection($community_posts);
        $response = generateResponse($community_posts, true, 'CommunityPosts Fetched Successfully', null, 'collection');
        return response()->json($response, 200);
    }

    /********* FILTER CommunityPosts FOR Search ***********/
    public function filter(Request $request)
    {
        $therapist_posts = $this->getter($request);
        $response = generateResponse($therapist_posts, count($therapist_posts['data']) > 0 ? true : false, 'Filter CommunityPosts Successfully', null, 'collection');
        return response()->json($response, 200);
    }

    /********* ADD NEW TherapistPost ***********/
    public function store(Request $request)
    {

        // dd($request->all());
        $therapist = auth()->user()->therapist;
        if($request->is_shared){
            try {
                DB::beginTransaction();
                $request->merge(['created_by_user_id' => auth()->user()->id,'description' => $request->sharedtext,'share_with' => $therapist->id]);
                $data = $request->all();
                if($request->sharedImg){
                    $data['image'] = $request->sharedImg;
                }
                $therapist_post = $therapist->therapist_posts()->create($data);
                $therapist_post->slug = Str::slug(Str::limit($therapist_post->description, 20) . '-' . $therapist_post->id);
                $therapist_post->save();
                $therapist_post = $therapist->therapist_posts()->withAll()->find($therapist_post->id);
                $therapist_post = new CommunityPostsResource($therapist_post);
                // $therapist_post->tags()->sync($request->tag_ids);
                DB::commit();
            } catch (\Exception $e) {
                DB::rollBack();
                request()->session()->flash('alert', ['message' => 'Invalid Request', 'type' => 'error']);
            }
            return redirect()->back();
        }
        $request->validate([

            'description' => ['required'],
            'attachment' => ['nullable', 'mimes:jpeg,png,jpg,mp4,ogx,oga,ogv,ogg,webm'],
        ], [
            'description.required' => 'The post description field is required.',
            'attachment.mimes' => 'The attachment must be a file of type: jpeg, png, jpg.',
        ]);
        try {
            DB::beginTransaction();
            $request->merge(['created_by_user_id' => auth()->user()->id]);

            $data = $request->all();

            if ($request->hasFile('attachment')) {
                $file = $request->file('attachment');
                $mimeType = $file->getClientMimeType();

                if (str_starts_with($mimeType, 'video/')) {
                    $data['video'] = uploadFile($request, 'attachment', 'community_posts/videos');
                } else if (str_starts_with($mimeType, 'image/')) {
                    $data['image'] = uploadFile($request, 'attachment', 'community_posts/images');
                }
            }

            // $data['image'] = uploadFile($request, 'attachment', 'community_posts');
            $therapist_post = $therapist->therapist_posts()->create($data);
            $therapist_post->slug = Str::slug(Str::limit($therapist_post->description, 20) . '-' . $therapist_post->id);
            $therapist_post->save();
            $therapist_post = $therapist->therapist_posts()->withAll()->find($therapist_post->id);
            $therapist_post = new CommunityPostsResource($therapist_post);
            // $therapist_post->tags()->sync($request->tag_ids);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            request()->session()->flash('alert', ['message' => 'Invalid Request', 'type' => 'error']);
        }
        return redirect()->back();
    }

    /********* View RECORD TO EDIT Or Display ***********/
    public function show($therapist_post)
    {
        $therapist = auth()->user()->therapist;
        if ($therapist_post->therapist_id != $therapist->id) {
            return redirect()->back()->withErrors([
                'message' => 'Invalid Request',
                'type' => 'error'
            ]);
        }
        $therapist_post = $therapist->therapist_posts()->withAll()->find($therapist_post);
        if ($therapist_post) {
            $therapist_post = new CommunityPostsResource($therapist_post);
            $response = generateResponse($therapist_post, true, 'TherapistPost Fetched Successfully', null, 'object');
        } else {
            $response = generateResponse(null, false, 'TherapistPost Not FOund', null, 'object');
        }
        return response()->json($response, 200);
    }

    /********* UPDATE TherapistPost ***********/
    public function update(CreateRequest $request, Post $therapist_post)
    {
        // dd($request->all());
        $therapist = auth()->user()->therapist;
        if ($therapist_post->therapist_id != $therapist->id) {
            return redirect()->back()->withErrors([
                'message' => 'Invalid Request',
                'type' => 'error'
            ]);
        }
        try {
            DB::beginTransaction();
            $request->merge(['last_updated_by_user_id' => auth()->user()->id]);
            $data = $request->all();
            if ($request->image) {
                $data['image'] = uploadCroppedFile($request, 'image', 'therapist_posts', $therapist_post->image);
            } else {
                $data['image'] = $therapist_post->image;
            }
            $therapist_post->update($data);
            $therapist_post = $therapist_post->find($therapist_post->id);
            $slug = Str::slug($therapist_post['name'] . ' ' . $therapist_post->id, '-');
            $therapist_post->update(
                [
                    'slug' => $slug
                ]
            );
            $therapist_post->tags()->sync($request->tag_ids);
            DB::commit();
        } catch (\Exception $e) {
            dd($e->getMessage());
            DB::rollBack();
            request()->session()->flash('alert', ['message' => 'Invalid Request', 'type' => 'error']);
        }
        return redirect()->back();
    }

    /********* UPDATE TherapistPost Status***********/
    public function updateStatus(Request $request, Post $therapist_post)
    {
        $therapist = auth()->user()->therapist;
        if ($therapist_post->therapist_id != $therapist->id) {
            return redirect()->back()->withErrors([
                'message' => 'Invalid Request',
                'type' => 'error'
            ]);
        }
        $therapist_post->update([
            'is_active' => $therapist_post->is_active == 1 ? 0 : 1
        ]);
        $response = generateResponse(null, true, 'TherapistPost Status Updated Successfully', null, 'object');
        return response()->json($response, 200);
    }


    /********* DELETE TherapistPost ***********/
    public function destroy(Request $request, Post $therapist_post)
    {
        $therapist = auth()->user()->therapist;
        if ($therapist_post->therapist_id != $therapist->id) {
            request()->session()->flash('alert', ['message' => 'Invalid Request', 'type' => 'error']);
            return redirect()->back();
        }
        if ($therapist_post->trashed()) {
            request()->session()->flash('alert', ['message' => 'Already in Trash', 'type' => 'error']);
        } else {
            $therapist_post->delete();
        }
        return redirect()->back();
    }
    /*********Permanently DELETE TherapistPost ***********/
    public function destroyPermanently(Request $request, $therapist_post)
    {
        $therapist = auth()->user()->therapist;
        $therapist_post = $therapist->therapist_posts()->withTrashed()->find($therapist_post);
        if ($therapist_post) {
            if ($therapist_post->therapist_id != $therapist->id) {
                return redirect()->back()->withErrors([
                    'message' => 'Invalid Request',
                    'type' => 'error'
                ]);
            }
            if ($therapist_post->trashed()) {
                $therapist_post->forceDelete();
                $response = generateResponse(null, true, 'TherapistPost Deleted Successfully', null, 'object');
            } else {
                $response = generateResponse(null, false, 'TherapistPost is not in trash to delete permanently', null, 'object');
            }
        } else {
            $response = generateResponse(null, false, 'TherapistPost not found', null, 'object');
        }
        return response()->json($response, 200);
    }
    /********* Restore TherapistPost ***********/
    public function restore(Request $request, $therapist_post)
    {
        $therapist = auth()->user()->therapist;
        $therapist_post = $therapist->therapist_posts()->withTrashed()->find($therapist_post);
        if ($therapist_post->trashed()) {
            $therapist_post->restore();
            $response = generateResponse(null, true, 'TherapistPost Restored Successfully', null, 'object');
        } else {
            $response = generateResponse(null, false, 'TherapistPost is not trashed', null, 'object');
        }
        return response()->json($response, 200);
    }

    public function postLike(Request $request, Post $post)
    {
        $therapist_id = $post->therapist_id ?? auth()->user()->id;

        $existing_like = $post->post_likes()->where('therapist_id', $therapist_id)->first();

        $new_is_like = $existing_like ? !$existing_like->is_like : 1;

        $community_post = $post->post_likes()->updateOrCreate(
            ['therapist_id' => $therapist_id],
            ['is_like' => $new_is_like]
        );

        $message = $new_is_like ? 'Post Liked Successfully' : 'Like Removed Successfully';
        $post = Post::where('id', $post->id)->withAll()->withChildren()->first();
        $post = new CommunityPostsResource($post);
        $response = generateResponse($post, true, $message, null, 'object');
        return response()->json($response, 200);
    }

    public function postComment(Request $request, Post $post)
    {
        $auth_id = auth()->user()->id;
        $therapist = auth()->user()->therapist;
        $request->merge(['user_id' => $auth_id, 'content' => $request->comment,'therapist_id' => $therapist?->id]);
        $post->post_comments()->create($request->all());
        $response = generateResponse(null, true, 'Comment Recorded Successfully', null, 'object');
        return response()->json($response, 200);
    }
    public function commentReply(Request $request, PostComment $comment)
    {

        $reply = new PostComment;
        $reply->post_id = $comment->post_id;
        $reply->user_id = $comment->user_id;
        $reply->content = $request->content;
        $reply->parent_id = $comment->id;
        $reply->save();


        // $request->merge(['post_id',$comment->post_id,'user_id',$comment->user_id]);
        // $comment->replies()->create($request->all());
        $response = generateResponse(null, true, 'Comment Reply Recorded Successfully', null, 'object');
        return response()->json($response, 200);
    }
}
